(function () {
  'use strict';

  // ------- helpers -------
  const $ = (sel) => document.querySelector(sel);
  const base = window.BASE_PATH || '';

  function apiUrl(path, params) {
    let url = base + path;
    if (params && Object.keys(params).length) {
      url += '?' + new URLSearchParams(params).toString();
    }
    return url;
  }

  async function buscar(q, source = 'text') {
    if (!q) return;
    results.innerHTML = '<div class="empty">Buscando…</div>';
    try {
      const url = apiUrl('/buscar', { q, source });
      const res = await fetch(url, {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        cache: 'no-store'
      });

      const ct = res.headers.get('content-type') || '';
      if (!ct.includes('application/json')) {
        const txt = await res.text();
        console.error('Respuesta no-JSON:', txt);
        render({ type: 'none', message: 'Error del servidor' });
        return;
      }

      const json = await res.json();
      render(json);
    } catch (e) {
      console.error(e);
      render({ type: 'none', message: 'No se pudo conectar' });
    }
  }

  async function loadUnitInline(slug) {
    results.innerHTML = '<div class="empty">Cargando…</div>';
    try {
      const url = apiUrl('/unidad/' + slug, { partial: '1' });
      const res = await fetch(url, {
        headers: { 'X-Requested-With': 'fetch' },
        cache: 'no-store'
      });
      const html = await res.text();
      results.innerHTML = html;

      const back = $('#btn-back');
      if (back) {
        back.onclick = () => {
          results.innerHTML = '';
          input.focus();
        };
      }
    } catch (e) {
      console.error(e);
      results.innerHTML = '<div class="empty">No se pudo cargar la unidad.</div>';
    }
  }

  function render(data) {
    results.innerHTML = '';

    if (data.type === 'single' && data.unit && data.unit.slug) {
      // ✅ Cargar el detalle dentro de #results sin navegar
      loadUnitInline(data.unit.slug);
      return;
    }

    if (data.type === 'multiple' && Array.isArray(data.results)) {
      const ul = document.createElement('ul');
      ul.className = 'list';
      data.results.forEach((r) => {
        const li = document.createElement('li');
        li.innerHTML = `
          <div>
            <strong>${r.name}</strong>
            ${r.alias ? `<span class="muted">(${r.alias})</span>` : ''}
          </div>
          <div>
            <a class="btn small" href="${base + '/unidad/' + r.slug}">Ver cómo llegar</a>
          </div>
        `;
        // Interceptamos el click para evitar navegación y cargar inline
        li.querySelector('a').addEventListener('click', (ev) => {
          ev.preventDefault();
          loadUnitInline(r.slug);
        });
        ul.appendChild(li);
      });
      results.innerHTML = `<h3>¿Te refieres a…?</h3>`;
      results.appendChild(ul);
      return;
    }

    if (data.type === 'none') {
      const sug = (data.suggestions || []).map((s) => `<code>${s}</code>`).join(' ');
      results.innerHTML =
        `<div class="empty">No encontré esa unidad. ${sug ? 'Prueba con: ' + sug : ''}</div>`;
      return;
    }

    if (data.type === 'error') {
      results.innerHTML = `<div class="empty">${data.message || 'Error'}</div>`;
      return;
    }

    results.innerHTML = `<div class="empty">Ocurrió un problema. Intenta de nuevo.</div>`;
  }

  // ------- elementos -------
  const btn = $('#btn-voice');
  const status = $('#voice-status');
  const form = $('#search-form');
  const input = $('#q');
  const results = $('#results');

  // ------- formulario (GET por AJAX) -------
  if (form) {
    form.addEventListener('submit', (e) => {
      e.preventDefault();
      const qv = input.value.trim();
      if (!qv) return;
      buscar(qv, 'text');
    });
  }

  // ------- Web Speech API (voz → GET AJAX) -------
  const SpeechRecognition = window.SpeechRecognition || window.webkitSpeechRecognition;

  if (!SpeechRecognition) {
    if (btn) btn.disabled = true;
    if (status) status.textContent = 'Voz no disponible en este navegador. Escribe tu consulta.';
  } else {
    const rec = new SpeechRecognition();
    rec.lang = 'es-ES';
    rec.interimResults = true;
    rec.continuous = false;

    let listening = false;
    let finalText = '';

    if (btn) {
      btn.addEventListener('click', () => {
        if (!listening) rec.start(); else rec.stop();
      });
    }

    rec.onstart = () => {
      listening = true;
      if (btn) btn.textContent = '🛑 Detener';
      if (status) status.textContent = 'Escuchando…';
    };

    rec.onend = () => {
      listening = false;
      if (btn) btn.textContent = '🎤 Presiona y habla';
      if (status) status.textContent = 'Micrófono inactivo';

      const q = finalText.trim();
      if (q) buscar(q, 'voice'); // lama /buscar por GET y renderiza en #results
      finalText = '';
    };

    rec.onresult = (ev) => {
      let interim = '';
      for (let i = ev.resultIndex; i < ev.results.length; i++) {
        const t = ev.results[i][0].transcript;
        if (ev.results[i].isFinal) finalText += t; else interim += t;
      }
      if (input) input.value = (finalText || interim).trim();
    };

    rec.onerror = (e) => {
      if (status) status.textContent = 'Error de voz: ' + (e.error || 'desconocido');
    };
  }

  // ------- auto-búsqueda si llega ?q=... en la URL (sin recargar) -------
  try {
    const params = new URLSearchParams(location.search);
    const initialQ = params.get('q');
    if (initialQ) {
      input.value = initialQ;
      buscar(initialQ, 'text');
    }
  } catch (_) { /* nop */ }
})();
