// public/js/admin-dashboard.js
// Dashboard Admin – JS V3 (contraste + tooltips + LKG/Demo)
const API_BASE = '/campusmap/admin/api/stats'; // ajusta a '/campusmap/admin/api/stats' si usas prefijo
const TZ = 'America/El_Salvador';
const CONCURRENCY_LIMIT = 5;

// ===== Texto de ayuda por KPI =====
const KPI_INFO = {
  searches: 'Total de consultas realizadas en el rango. Útil para ver demanda/uso.',
  noResult: 'Porcentaje de búsquedas sin coincidencia. Si supera 15% revisar catálogo y sinónimos.',
  topUnits: 'Unidades más consultadas. Prioriza contenido/fotos de estas.',
  funnel: 'Embudo: Búsqueda → Unidad vista → Ruta solicitada. Observa dónde se pierde.',
  ctr: 'CTR estimado de búsqueda a selección. <40% suele indicar relevancia baja.',
  routeRequests: 'Veces que los usuarios solicitaron la ruta (play/qr). Indicador de conversión.',
  kioskSessions: 'Sesiones iniciadas en el kiosko (hoy/rango). Refleja afluencia.',
  catalogHygiene: 'Higiene del catálogo: unidades sin media/sin sinónimos/sin metadatos.',
  security: 'Usuarios del panel actualmente bloqueados por intentos fallidos.',
  usageByChannel: 'Distribución de uso por canal (kiosk/web/mobile).',
  inputMode: 'Modo de entrada en búsquedas (texto/voz).'
};

// ===== Utilidades de fecha/rango =====
function todayISO(tz = TZ) {
  const now = new Date();
  const local = new Date(now.toLocaleString('en-US', { timeZone: tz }));
  const y = local.getFullYear();
  const m = String(local.getMonth() + 1).padStart(2, '0');
  const d = String(local.getDate()).padStart(2, '0');
  return `${y}-${m}-${d}`;
}
function readRangeFromUI() {
  const startEl = document.getElementById('start-date');
  const endEl   = document.getElementById('end-date');
  let start = startEl?.value || todayISO();
  let end   = endEl?.value   || todayISO();
  if (start > end) [start, end] = [end, start];
  return { start, end };
}
function qs(params) { return `?${new URLSearchParams(params).toString()}`; }

// ===== Helpers DOM/Render =====
function body(el){ return el?.querySelector('.card-body') || el; }
function setSkeleton(el, isLoading = true) {
  const target = body(el);
  if (!target) return;
  target.classList.toggle('is-loading', isLoading);
  if (isLoading) target.innerHTML = `<div class="skeleton"></div>`;
}
function safeText(s) {
  return String(s ?? '').replace(/[&<>"'`]/g, ch =>
    ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;','`':'&#96;'}[ch])
  );
}
function renderError(el, msg = 'Sin datos') {
  const target = body(el);
  if (!target) return;
  target.classList.remove('is-loading');
  target.innerHTML = `<div class="card-error">${safeText(msg)}</div>`;
}
function formatPct(n, digits = 1) { return (n == null || isNaN(n)) ? '0%' : `${Number(n).toFixed(digits)}%`; }
function formatInt(n) { return (n == null || isNaN(n)) ? '0' : Number(n).toLocaleString('es-SV'); }

// Botón de info (tooltip puro CSS con attr)
function infoBtn(key){
  const tip = KPI_INFO[key];
  if (!tip) return '';
  return `<button class="kpi-info" aria-label="info" data-tip="${safeText(tip)}">i</button>`;
}

// ===== Fetch con reintentos y límite de concurrencia =====
let activeFetches = 0; const queue = [];
async function schedule(task) {
  if (activeFetches >= CONCURRENCY_LIMIT) await new Promise(res => queue.push(res));
  activeFetches++;
  try { return await task(); }
  finally { activeFetches--; const next = queue.shift(); if (next) next(); }
}
async function fetchJSON(url, opts = {}, retries = 2) {
  const attempt = async (delayMs) => {
    if (delayMs) await new Promise(r => setTimeout(r, delayMs));
    const resp = await fetch(url, { credentials: 'same-origin', ...opts });
    if (!resp.ok) throw new Error(`HTTP ${resp.status}`);
    const data = await resp.json();
    if (!data || data.ok === false) throw new Error(data?.error || 'Respuesta inválida');
    return { data };
  };
  try { return await attempt(0); }
  catch (e) { if (retries <= 0) throw e; return attempt((3 - retries) * 400 + 200); }
}

// ===== Endpoints =====
const endpoints = {
  searches:        p => `${API_BASE}/searches${qs(p)}`,
  noResult:        p => `${API_BASE}/no-result${qs(p)}`,
  topUnits:        p => `${API_BASE}/top-units${qs(p)}`,
  funnel:          p => `${API_BASE}/funnel${qs(p)}`,
  ctr:             p => `${API_BASE}/ctr${qs(p)}`,
  routeRequests:   p => `${API_BASE}/route-requests${qs(p)}`,
  kioskSessions:   p => `${API_BASE}/kiosk-sessions${qs(p)}`,
  catalogHygiene:  p => `${API_BASE}/catalog-hygiene${qs(p)}`,
  security:        p => `${API_BASE}/security${qs(p)}`,
  usageByChannel:  p => `${API_BASE}/usage-by-channel${qs(p)}`,
  inputMode:       p => `${API_BASE}/input-mode${qs(p)}`,
};

// ===== LKG + Demo =====
const DEMO_FALLBACK = true;
function saveLastGood(key, payload) { try { localStorage.setItem(`dash:lkg:${key}`, JSON.stringify({t:Date.now(), payload})); } catch {} }
function loadLastGood(key, maxAgeMin = 1440) {
  try {
    const raw = localStorage.getItem(`dash:lkg:${key}`); if (!raw) return null;
    const obj = JSON.parse(raw); if (!obj?.t) return null;
    const age = (Date.now() - obj.t) / 60000; if (age > maxAgeMin) return null;
    return obj.payload;
  } catch { return null; }
}
const DEMO = {
  searches:        { count: 128, series: [{date:'2025-10-15',count:20},{date:'2025-10-16',count:22},{date:'2025-10-17',count:18},{date:'2025-10-18',count:25},{date:'2025-10-19',count:43}] },
  noResult:        { no_result_pct: 9.8 },
  topUnits:        { items: [{name:'Registro Académico',hits:32},{name:'Caja',hits:27},{name:'Biblioteca',hits:22},{name:'Nuevo Ingreso',hits:19},{name:'R.R.H.H.',hits:11}] },
  funnel:          { searches: 200, unit_views: 120, route_requests: 48 },
  ctr:             { ctr_search_to_selection_pct: 60.0 },
  routeRequests:   { count: 48 },
  kioskSessions:   { count: 34 },
  catalogHygiene:  { without_media: 8, without_synonyms: 15, missing_meta: 3 },
  security:        { locked_count: 0, locked_list: [] },
  usageByChannel:  { items: [{source:'kiosk',events:210},{source:'web',events:40},{source:'mobile',events:18}] },
  inputMode:       { items: [{mode:'text',searches:270},{mode:'voice',searches:30}] },
};
function renderFallback(el, key, renderFn) {
  const lkg = loadLastGood(key);
  const src = lkg || (DEMO_FALLBACK ? DEMO[key] : null);
  if (!src) { renderError(el, 'Sin datos.'); return; }
  const target = body(el); el?.classList?.add('demo-fallback');
  renderFn(src, !lkg);
  if (!lkg) {
    const title = target.querySelector('.kpi-title');
    if (title && !title.innerHTML.includes('Demo')) {
      title.innerHTML += ` <span class="badge bg-secondary ms-2">Demo</span>`;
    }
  }
}

// ===== Renderizadores (con infoBtn + LKG/Demo) =====
async function loadSearches(params) {
  const el = document.getElementById('card-searches'); if (!el) return; setSkeleton(el,true);
  try {
    const { data } = await schedule(() => fetchJSON(endpoints.searches(params)));
    const p = data.data || {}; saveLastGood('searches', p);
    const target = body(el); target.classList.remove('is-loading');
    const trend = (p.series ?? []).slice(-7).map(s => s.count).join(' · ');
    target.innerHTML = `
      <div class="kpi">
        <div class="kpi-head"><div class="kpi-title">Búsquedas</div>${infoBtn('searches')}</div>
        <div class="kpi-value">${formatInt(p.count ?? 0)}</div>
        <div class="kpi-sub">Últimos días: ${safeText(trend)}</div>
      </div>`;
  } catch {
    renderFallback(el, 'searches', (p) => {
      const target = body(el); target.classList.remove('is-loading');
      const trend = (p.series ?? []).slice(-7).map(s => s.count).join(' · ');
      target.innerHTML = `
        <div class="kpi">
          <div class="kpi-head"><div class="kpi-title">Búsquedas</div>${infoBtn('searches')}</div>
          <div class="kpi-value">${formatInt(p.count ?? 0)}</div>
          <div class="kpi-sub">Últimos días: ${safeText(trend)}</div>
        </div>`;
    });
  }
}
async function loadNoResult(params) {
  const el = document.getElementById('card-no-result'); if (!el) return; setSkeleton(el,true);
  try {
    const { data } = await schedule(() => fetchJSON(endpoints.noResult(params)));
    const p = data.data || {}; saveLastGood('noResult', p);
    const target = body(el); const pct = p.no_result_pct ?? 0; const alert = pct > 15 ? 'warn':'';
    target.classList.remove('is-loading');
    target.innerHTML = `
      <div class="kpi ${alert}">
        <div class="kpi-head"><div class="kpi-title">% búsquedas sin resultado</div>${infoBtn('noResult')}</div>
        <div class="kpi-value">${formatPct(pct, 1)}</div>
        <div class="kpi-sub">${pct > 15 ? '⚠ Revisa sinónimos y catálogo' : '&nbsp;'}</div>
      </div>`;
  } catch {
    renderFallback(el, 'noResult', (p) => {
      const target = body(el); const pct = p.no_result_pct ?? 0; const alert = pct > 15 ? 'warn':'';
      target.classList.remove('is-loading');
      target.innerHTML = `
        <div class="kpi ${alert}">
          <div class="kpi-head"><div class="kpi-title">% búsquedas sin resultado</div>${infoBtn('noResult')}</div>
          <div class="kpi-value">${formatPct(pct, 1)}</div>
          <div class="kpi-sub">${pct > 15 ? '⚠ Revisa sinónimos y catálogo' : '&nbsp;'}</div>
        </div>`;
    });
  }
}
async function loadTopUnits(params) {
  const el = document.getElementById('card-top-units'); if (!el) return; setSkeleton(el,true);
  try {
    const { data } = await schedule(() => fetchJSON(endpoints.topUnits(params)));
    const p = data.data || {}; saveLastGood('topUnits', p);
    const items = p.items ?? []; const target = body(el); target.classList.remove('is-loading');
    target.innerHTML = `
      <div class="kpi">
        <div class="kpi-head"><div class="kpi-title">Top unidades buscadas</div>${infoBtn('topUnits')}</div>
        <div class="kpi-list">
          ${items.slice(0,5).map(it => `
            <div class="row"><span class="name">${safeText(it.name)}</span><span class="val">${formatInt(it.hits)}</span></div>`).join('')}
        </div>
      </div>`;
  } catch {
    renderFallback(el, 'topUnits', (p) => {
      const items = p.items ?? []; const target = body(el); target.classList.remove('is-loading');
      target.innerHTML = `
        <div class="kpi">
          <div class="kpi-head"><div class="kpi-title">Top unidades buscadas</div>${infoBtn('topUnits')}</div>
          <div class="kpi-list">
            ${items.slice(0,5).map(it => `
              <div class="row"><span class="name">${safeText(it.name)}</span><span class="val">${formatInt(it.hits)}</span></div>`).join('')}
          </div>
        </div>`;
    });
  }
}
async function loadFunnel(params) {
  const el = document.getElementById('card-funnel');
  if (!el) return;
  setSkeleton(el, true);
  
  try {
    const { data } = await schedule(() => fetchJSON(endpoints.funnel(params)));
    const p = data.data || {};
    saveLastGood('funnel', p);
    
    const { searches = 0, unit_views = 0, route_requests = 0 } = p;

    // Evita porcentajes absurdos
    const pct1 = searches ? Math.min(100, (unit_views / searches) * 100) : 0;
    const pct2 = unit_views ? Math.min(100, (route_requests / unit_views) * 100) : 0;

    const target = body(el);
    target.classList.remove('is-loading');
    target.innerHTML = `
      <div class="kpi">
        <div class="kpi-head"><div class="kpi-title">Embudo</div>${infoBtn('funnel')}</div>
        <div class="funnel">
          <div class="stage">
            <div class="label">Búsquedas</div>
            <div class="value">${formatInt(searches)}</div>
          </div>
          <div class="stage">
            <div class="label">Unidad vista</div>
            <div class="value">${formatInt(unit_views)} <span class="muted">(${formatPct(pct1)})</span></div>
          </div>
          <div class="stage">
            <div class="label">Ruta solicitada</div>
            <div class="value">${formatInt(route_requests)} <span class="muted">(${formatPct(pct2)})</span></div>
          </div>
        </div>
      </div>`;
  } catch {
    renderFallback(el, 'funnel', (p) => {
      const { searches = 0, unit_views = 0, route_requests = 0 } = p;
      const pct1 = searches ? Math.min(100, (unit_views / searches) * 100) : 0;
      const pct2 = unit_views ? Math.min(100, (route_requests / unit_views) * 100) : 0;
      const target = body(el);
      target.classList.remove('is-loading');
      target.innerHTML = `
        <div class="kpi">
          <div class="kpi-head"><div class="kpi-title">Embudo</div>${infoBtn('funnel')}</div>
          <div class="funnel">
            <div class="stage"><div class="label">Búsquedas</div><div class="value">${formatInt(searches)}</div></div>
            <div class="stage"><div class="label">Unidad vista</div><div class="value">${formatInt(unit_views)} <span class="muted">(${formatPct(pct1)})</span></div></div>
            <div class="stage"><div class="label">Ruta solicitada</div><div class="value">${formatInt(route_requests)} <span class="muted">(${formatPct(pct2)})</span></div></div>
          </div>
        </div>`;
    });
  }
}

async function loadCTR(params) {
  const el = document.getElementById('card-ctr'); 
  if (!el) return; 
  setSkeleton(el, true);

  try {
    const resp = await schedule(() => fetchJSON(endpoints.ctr(params)));
    console.log('CTR raw response:', resp);

    // ⚙️ Accedemos al nivel correcto (doble data)
    const p = resp.data?.data || {};
    console.log('CTR parsed data:', p);

    const pct = p.ctr_search_to_selection_pct ?? 0;
    const warn = pct < 40 ? 'warn' : '';
    const target = body(el);
    target.classList.remove('is-loading');
    target.innerHTML = `
      <div class="kpi ${warn}">
        <div class="kpi-head"><div class="kpi-title">CTR búsqueda → selección</div>${infoBtn('ctr')}</div>
        <div class="kpi-value">${formatPct(pct)}</div>
        <div class="kpi-sub">${pct < 40 ? '⚠ Mejorar relevancia de resultados' : '&nbsp;'}</div>
      </div>`;
  } catch (err) {
    console.error('Error CTR:', err);
    renderFallback(el, 'ctr', (p) => {
      const pct = p.ctr_search_to_selection_pct ?? 0;
      const warn = pct < 40 ? 'warn' : '';
      const target = body(el);
      target.classList.remove('is-loading');
      target.innerHTML = `
        <div class="kpi ${warn}">
          <div class="kpi-head"><div class="kpi-title">CTR búsqueda → selección</div>${infoBtn('ctr')}</div>
          <div class="kpi-value">${formatPct(pct)}</div>
          <div class="kpi-sub">${pct < 40 ? '⚠ Mejorar relevancia de resultados' : '&nbsp;'}</div>
        </div>`;
    });
  }
}


async function loadRouteRequests(params) {
  const el = document.getElementById('card-route-requests'); if (!el) return; setSkeleton(el,true);
  try {
    const { data } = await schedule(() => fetchJSON(endpoints.routeRequests(params)));
    const p = data.data || {}; saveLastGood('routeRequests', p);
    const target = body(el); target.classList.remove('is-loading');
    target.innerHTML = `
      <div class="kpi">
        <div class="kpi-head"><div class="kpi-title">Solicitudes de ruta</div>${infoBtn('routeRequests')}</div>
        <div class="kpi-value">${formatInt(p.count ?? 0)}</div>
      </div>`;
  } catch {
    renderFallback(el, 'routeRequests', (p) => {
      const target = body(el); target.classList.remove('is-loading');
      target.innerHTML = `
        <div class="kpi">
          <div class="kpi-head"><div class="kpi-title">Solicitudes de ruta</div>${infoBtn('routeRequests')}</div>
          <div class="kpi-value">${formatInt(p.count ?? 0)}</div>
        </div>`;
    });
  }
}
async function loadKioskSessions(params) {
  const el = document.getElementById('card-kiosk-sessions'); if (!el) return; setSkeleton(el,true);
  try {
    const { data } = await schedule(() => fetchJSON(endpoints.kioskSessions(params)));
    const p = data.data || {}; saveLastGood('kioskSessions', p);
    const target = body(el); target.classList.remove('is-loading');
    target.innerHTML = `
      <div class="kpi">
        <div class="kpi-head"><div class="kpi-title">Sesiones de kiosko</div>${infoBtn('kioskSessions')}</div>
        <div class="kpi-value">${formatInt(p.count ?? 0)}</div>
      </div>`;
  } catch {
    renderFallback(el, 'kioskSessions', (p) => {
      const target = body(el); target.classList.remove('is-loading');
      target.innerHTML = `
        <div class="kpi">
          <div class="kpi-head"><div class="kpi-title">Sesiones de kiosko</div>${infoBtn('kioskSessions')}</div>
          <div class="kpi-value">${formatInt(p.count ?? 0)}</div>
        </div>`;
    });
  }
}
async function loadCatalogHygiene(params) {
  const el = document.getElementById('card-catalog-hygiene'); if (!el) return; setSkeleton(el,true);
  try {
    const { data } = await schedule(() => fetchJSON(endpoints.catalogHygiene(params)));
    const p = data.data || {}; saveLastGood('catalogHygiene', p);
    const { without_media=0, without_synonyms=0, missing_meta=0 } = p;
    const target = body(el); target.classList.remove('is-loading');
    target.innerHTML = `
      <div class="kpi">
        <div class="kpi-head"><div class="kpi-title">Higiene del catálogo</div>${infoBtn('catalogHygiene')}</div>
        <div class="kpi-grid">
          <div><div class="mini-title">Sin media</div><div class="mini-val">${formatInt(without_media)}</div></div>
          <div><div class="mini-title">Sin sinónimos</div><div class="mini-val">${formatInt(without_synonyms)}</div></div>
          <div><div class="mini-title">Meta faltante</div><div class="mini-val">${formatInt(missing_meta)}</div></div>
        </div>
      </div>`;
  } catch {
    renderFallback(el, 'catalogHygiene', (p) => {
      const { without_media=0, without_synonyms=0, missing_meta=0 } = p;
      const target = body(el); target.classList.remove('is-loading');
      target.innerHTML = `
        <div class="kpi">
          <div class="kpi-head"><div class="kpi-title">Salud del catálogo</div>${infoBtn('catalogHygiene')}</div>
          <div class="kpi-grid">
            <div><div class="mini-title">Sin media</div><div class="mini-val">${formatInt(without_media)}</div></div>
            <div><div class="mini-title">Sin sinónimos</div><div class="mini-val">${formatInt(without_synonyms)}</div></div>
            <div><div class="mini-title">Meta faltante</div><div class="mini-val">${formatInt(missing_meta)}</div></div>
          </div>
        </div>`;
    });
  }
}
async function loadSecurity(params) {
  const el = document.getElementById('card-security'); if (!el) return; setSkeleton(el,true);
  try {
    const { data } = await schedule(() => fetchJSON(endpoints.security(params)));
    const p = data.data || {}; saveLastGood('security', p);
    const locked = p.locked_count ?? 0; const list = p.locked_list ?? []; const alert = locked > 0 ? 'warn':'';
    const target = body(el); target.classList.remove('is-loading');
    target.innerHTML = `
      <div class="kpi ${alert}">
        <div class="kpi-head"><div class="kpi-title">Seguridad (bloqueos)</div>${infoBtn('security')}</div>
        <div class="kpi-value">${formatInt(locked)}</div>
        <div class="kpi-sub">${locked>0 ? 'Usuarios bloqueados activos' : '&nbsp;'}</div>
        ${locked>0 ? `<div class="kpi-list small">
          ${list.slice(0,5).map(u => `
            <div class="row"><span class="name">${safeText(u.email || '—')}</span><span class="val">${safeText(u.locked_until || '')}</span></div>`).join('')}
        </div>`:''}
      </div>`;
  } catch {
    renderFallback(el, 'security', (p) => {
      const locked = p.locked_count ?? 0; const list = p.locked_list ?? []; const alert = locked > 0 ? 'warn':'';
      const target = body(el); target.classList.remove('is-loading');
      target.innerHTML = `
        <div class="kpi ${alert}">
          <div class="kpi-head"><div class="kpi-title">Seguridad (bloqueos)</div>${infoBtn('security')}</div>
          <div class="kpi-value">${formatInt(locked)}</div>
          <div class="kpi-sub">${locked>0 ? 'Usuarios bloqueados activos' : '&nbsp;'}</div>
          ${locked>0 ? `<div class="kpi-list small">
            ${list.slice(0,5).map(u => `
              <div class="row"><span class="name">${safeText(u.email || '—')}</span><span class="val">${safeText(u.locked_until || '')}</span></div>`).join('')}
          </div>`:''}
        </div>`;
    });
  }
}
async function loadUsageByChannel(params) {
  const el = document.getElementById('card-usage-by-channel'); if (!el) return; setSkeleton(el,true);
  try {
    const { data } = await schedule(() => fetchJSON(endpoints.usageByChannel(params)));
    const p = data.data || {}; saveLastGood('usageByChannel', p);
    const items = p.items ?? []; const total = items.reduce((a,b)=>a+(b.events||0),0) || 1;
    const target = body(el); target.classList.remove('is-loading');
    target.innerHTML = `
      <div class="kpi">
        <div class="kpi-head"><div class="kpi-title">Uso por canal</div>${infoBtn('usageByChannel')}</div>
        <div class="bars">
          ${items.map(it=>{
            const pct = Math.round((it.events||0)*100/total);
            return `<div class="bar">
              <div class="bar-label">${safeText(it.source)}</div>
              <div class="bar-track"><div class="bar-fill" style="width:${pct}%"></div></div>
              <div class="bar-val">${formatInt(it.events)} (${pct}%)</div>
            </div>`;
          }).join('')}
        </div>
      </div>`;
  } catch {
    renderFallback(el, 'usageByChannel', (p) => {
      const items = p.items ?? []; const total = items.reduce((a,b)=>a+(b.events||0),0) || 1;
      const target = body(el); target.classList.remove('is-loading');
      target.innerHTML = `
        <div class="kpi">
          <div class="kpi-head"><div class="kpi-title">Uso por canal</div>${infoBtn('usageByChannel')}</div>
          <div class="bars">
            ${items.map(it=>{
              const pct = Math.round((it.events||0)*100/total);
              return `<div class="bar">
                <div class="bar-label">${safeText(it.source)}</div>
                <div class="bar-track"><div class="bar-fill" style="width:${pct}%"></div></div>
                <div class="bar-val">${formatInt(it.events)} (${pct}%)</div>
              </div>`;
            }).join('')}
          </div>
        </div>`;
    });
  }
}
async function loadInputMode(params) {
  const el = document.getElementById('card-input-mode'); if (!el) return; setSkeleton(el,true);
  try {
    const { data } = await schedule(() => fetchJSON(endpoints.inputMode(params)));
    const p = data.data || {}; saveLastGood('inputMode', p);
    const items = p.items ?? []; const total = items.reduce((a,b)=>a+(b.searches||0),0) || 1;
    const target = body(el); target.classList.remove('is-loading');
    target.innerHTML = `
      <div class="kpi">
        <div class="kpi-head"><div class="kpi-title">Modo de entrada</div>${infoBtn('inputMode')}</div>
        <div class="bars">
          ${items.map(it=>{
            const pct = Math.round((it.searches||0)*100/total);
            return `<div class="bar">
              <div class="bar-label">${safeText(it.mode)}</div>
              <div class="bar-track"><div class="bar-fill" style="width:${pct}%"></div></div>
              <div class="bar-val">${formatInt(it.searches)} (${pct}%)</div>
            </div>`;
          }).join('')}
        </div>
      </div>`;
  } catch {
    renderFallback(el, 'inputMode', (p) => {
      const items = p.items ?? []; const total = items.reduce((a,b)=>a+(b.searches||0),0) || 1;
      const target = body(el); target.classList.remove('is-loading');
      target.innerHTML = `
        <div class="kpi">
          <div class="kpi-head"><div class="kpi-title">Modo de entrada</div>${infoBtn('inputMode')}</div>
          <div class="bars">
            ${items.map(it=>{
              const pct = Math.round((it.searches||0)*100/total);
              return `<div class="bar">
                <div class="bar-label">${safeText(it.mode)}</div>
                <div class="bar-track"><div class="bar-fill" style="width:${pct}%"></div></div>
                <div class="bar-val">${formatInt(it.searches)} (${pct}%)</div>
              </div>`;
            }).join('')}
          </div>
        </div>`;
    });
  }
}

// ===== Batch opcional =====
async function tryBatchLoad(params) {
  const ids = [
    'card-searches','card-no-result','card-top-units','card-funnel','card-ctr',
    'card-route-requests','card-kiosk-sessions','card-catalog-hygiene','card-security',
    'card-usage-by-channel','card-input-mode'
  ];
  const present = ids.filter(id => document.getElementById(id));
  if (present.length === 0) return false;
  const url = `${API_BASE}/batch${qs({ ...params, metrics: present.join(',') })}`;
  try {
    const { data } = await fetchJSON(url, {}, 0);
    const map = data.data || {};
    if (!map || typeof map !== 'object') return false;
    return false; // si implementas batch, aquí haces el mapeo por card
  } catch { return false; }
}

// ===== Orquestación =====
async function loadAll() {
  const params = readRangeFromUI();
  [
    'card-searches','card-no-result','card-top-units','card-funnel','card-ctr',
    'card-route-requests','card-kiosk-sessions','card-catalog-hygiene','card-security',
    'card-usage-by-channel','card-input-mode'
  ].forEach(id => { const el = document.getElementById(id); if (el) setSkeleton(el,true); });
  const usedBatch = await tryBatchLoad(params); if (usedBatch) return;
  await Promise.all([
    () => loadSearches(params), () => loadNoResult(params), () => loadTopUnits(params),
    () => loadFunnel(params), () => loadCTR(params), () => loadRouteRequests(params),
    () => loadKioskSessions(params), () => loadCatalogHygiene(params), () => loadSecurity(params),
    () => loadUsageByChannel(params), () => loadInputMode(params)
  ].map(fn => fn()));
}

// ===== Estilos (contraste + tooltips) =====
(function injectStyles(){
  const style = document.createElement('style');
  style.innerHTML = `
    .is-loading .skeleton { height: 64px; border-radius: 12px; background: linear-gradient(90deg,#1b1b1b,#2a2a2a,#1b1b1b); background-size:200% 100%; animation: sh 1.2s infinite; }
    @keyframes sh { 0%{background-position:200% 0} 100%{background-position:-200% 0} }
    .kpi { padding: 12px; }
    .kpi-head { display:flex; align-items:center; justify-content:space-between; gap:8px; }
    .kpi-title { font-size:.95rem; color:#d0d0d0; margin-bottom:6px; letter-spacing:.2px; }
    .kpi-value { font-size:2rem; font-weight:800; line-height:1.1; color:#f2f2f2; }
    .kpi-sub { font-size:.85rem; color:#b8b8b8; margin-top:4px; }
    .kpi .muted { color:#b0b0b0; font-weight:600; margin-left:6px; }
    .kpi.warn .kpi-value { color:#ffa24c; }
    .kpi-list .row { display:flex; justify-content:space-between; padding:6px 0; border-bottom:1px dashed rgba(255,255,255,.06); color:#e6e6e6; }
    .kpi-list .row:last-child{ border-bottom:0; }
    .kpi-list .name{ max-width:70%; white-space:nowrap; overflow:hidden; text-overflow:ellipsis; color:#dcdcdc; }
    .kpi-grid { display:grid; grid-template-columns: repeat(3, 1fr); gap:8px; }
    .mini-title { font-size:.8rem; color:#b8b8b8; }
    .mini-val { font-size:1.15rem; font-weight:700; color:#efefef; }
    .funnel .stage { margin:8px 0; padding:6px 8px; background:#171717; border:1px solid #262626; border-radius:10px; }
    .funnel .label { font-size:.85rem; color:#b0b0b0; }
    .funnel .value { font-size:1.05rem; font-weight:700; color:#efefef; }
    .bars .bar { margin: 8px 0; color:#efefef; }
    .bar-label { font-size:.85rem; color:#b0b0b0; margin-bottom:4px; }
    .bar-track { height:10px; background:#2a2a2a; border-radius:6px; overflow:hidden; }
    .bar-fill { height:10px; background:linear-gradient(90deg,#9e9e9e,#d0d0d0); }
    .bar-val { font-size:.85rem; color:#dcdcdc; margin-top:4px; }
    .card-error { font-size:.92rem; color:#ffb4b4; background:#361e1e; border:1px solid #5a2d2d; padding:10px; border-radius:10px; }
    .demo-fallback { opacity: .97; }
    /* Botón de info + tooltip */
    .kpi-info {
      border:1px solid #3a3a3a; background:#1b1b1b; color:#e6e6e6; width:22px; height:22px;
      z-index: 9999;
      border-radius:50%; font-size:.8rem; line-height:1; display:inline-flex; align-items:center; justify-content:center;
      cursor:help; position:relative;
    }
    .kpi-info:hover { background:#242424; }
    .kpi-info::after {
      content: attr(data-tip);
      position:absolute; left:50%; transform:translate(-50%,-6px) translateY(-8px);
      opacity:0; pointer-events:none; transition:.15s ease; z-index:20;
      width:260px; max-width:70vw; padding:8px 10px; border-radius:8px;
      background:#111; border:1px solid #333; color:#e6e6e6; font-size:.82rem; text-align:left; white-space:normal;
    }
    .kpi-info:hover::after { opacity:1; transform:translate(-50%,-6px) translateY(-12px); }
    /* 1) Permite que el contenido pueda sobresalir */
.container-fluid,
.row,
[class^="col-"], [class*=" col-"],
.card,
.card .card-body {
  overflow: visible !important;
}

/* 2) Asegura stacking correcto de la card activa */
.card { position: relative; z-index: 1; }
.card:hover { z-index: 5; }

/* 3) El botón de info crea su propio contexto y eleva el tooltip */
.kpi-info { position: relative; z-index: 10; }
.kpi-info::after {
  z-index: 99999;                       /* muy por encima de otras capas */
  filter: drop-shadow(0 6px 12px rgba(0,0,0,.35));
}

/* (Opcional) si usas algún contenedor principal con overflow oculto, libéralo */
.main-content, .page-wrapper, .content-wrapper { overflow: visible !important; }

  `;
  document.head.appendChild(style);
})();

// ===== Init =====
document.addEventListener('DOMContentLoaded', () => {
  const btn = document.getElementById('apply-range');
  if (btn) btn.addEventListener('click', () => loadAll());
  loadAll();
});
