<?php
namespace App\Services;
use App\Models\UnitRepository;
use App\Models\SynonymRepository;
use App\Core\Database;
use PDO;

class SearchService {
  private UnitRepository $units;
  private SynonymRepository $syns;
  private PDO $db;

  public function __construct(private array $config){
    $this->units = new UnitRepository($config);
    $this->syns  = new SynonymRepository($config);
    $this->db    = Database::pdo($config);
  }

  public function match(string $q, string $source='text'): array {
    $raw = $q;
    $n = NLP::normalize($q);
    $n = NLP::removeStopwords($n);

    // 1) Sinónimo exacto
    $syn = $this->syns->findExact($n);
    if ($syn) {
      $this->log($raw, $syn['id'], 0.95, $source);
      return ['type'=>'single','unit'=>$syn];
    }

    // 2) Exacto en name/alias
    $ex = $this->units->exactByNameOrAlias($n);
    if ($ex) {
      $this->log($raw, $ex['id'], 0.9, $source);
      return ['type'=>'single','unit'=>$ex];
    }

    // 3) FULLTEXT
    $ft = $this->units->fulltext($n ?: $raw, 5);
    if ($ft) {
      // Si el primero se despega, ir directo; si no, desambiguar
      if (count($ft) === 1 || ($ft[0]['score'] >= 3 && ($ft[0]['score'] - ($ft[1]['score'] ?? 0)) > 1)) {
        $this->log($raw, (int)$ft[0]['id'], 0.8, $source);
        return ['type'=>'single','unit'=>[
          'id'=>$ft[0]['id'], 'name'=>$ft[0]['name'], 'alias'=>$ft[0]['alias'], 'slug'=>$ft[0]['slug']
        ]];
      }
      $this->log($raw, null, 0.5, $source);
      return ['type'=>'multiple','results'=>array_map(fn($r)=>[
        'id'=>$r['id'],'name'=>$r['name'],'alias'=>$r['alias'],'slug'=>$r['slug'],'score'=>$r['score']
      ], $ft)];
    }

    // 4) Sugerencias por LIKE de sinónimos
    $suggest = $n ? $this->syns->likeTerms($n, 5) : [];
    $this->log($raw, null, 0.0, $source);
    return ['type'=>'none','suggestions'=>array_column($suggest, 'term')];
  }

  private function log(string $q, ?int $unitId, float $conf, string $source): void {
    $st = $this->db->prepare("
      INSERT INTO search_logs (query_text, matched_unit_id, confidence, source, created_at)
      VALUES (?,?,?,?,NOW())
    ");
    $st->execute([$q, $unitId, $conf, $source]);
  }
}
