<?php
namespace App\Services;

use App\Models\KioskRepository;
use InvalidArgumentException;

class KioskService {
    private KioskRepository $repo;
    private UnitService $units;

    public function __construct(private array $config) {
        $this->repo  = new KioskRepository($config);
        $this->units = new UnitService($config); // aquí instanciamos UnitService
    }

    /**
     * Inicia una sesión de kiosko
     */
    public function startSession(string $role): string {
        if (!in_array($role, ['visitante', 'empleado', 'estudiante'], true)) {
            throw new InvalidArgumentException("Rol inválido");
        }

        $token = bin2hex(random_bytes(16));
        $this->repo->createSession($role, $token);
        $this->repo->logAction($token, 'voice_start');

        return $token;
    }

    /**
     * Procesa comando de voz
     */
    public function handleVoice(string $token, string $text): array {
        $search = new SearchService($this->config);
        return $search->match($text, 'voice');
    }

    /**
     * Registrar escaneo QR dentro de una sesión
     */
    public function logQrScan(string $token, int $unitId): void {
        $this->repo->logAction($token, "qr_scan:" . $unitId);
    }

    /**
     * Registrar acceso a detalle (cuando abre la unidad en la tableta)
     */
    public function logUnitAccess(string $token, int $unitId): void {
        $this->repo->logAction($token, "unit_access:" . $unitId);
    }

    /**
     * Registrar cualquier acción
     */
    public function logAction(string $session, string $action, ?int $unitId = null): void {
        $this->repo->logAction($session, $action, $unitId);
    }

    /**
     * Obtener detalle de una unidad
     */
    public function getUnitDetail($unitId = null, $slug = null): ?array {
        return $this->units->getUnitDetail($unitId, $slug);
    }
    
    public function getUnitSummary(int $unitId): ?array {
    $unitService = new UnitService($this->config);
    return $unitService->getUnitSummary($unitId);
}

}
