<?php
namespace App\Services\Admin;

use App\Models\Admin\SearchLogRepository;
use App\Models\Admin\UnitStatsRepository;
use App\Models\Admin\RouteRequestRepository;
use App\Models\Admin\KioskSessionRepository;
use App\Models\Admin\CatalogRepository;
use App\Models\Admin\SecurityRepository;
use App\Models\Admin\AnalyticsRepository;

/**
 * DashboardService
 * - Capa de dominio: agrega y transforma datos para el dashboard.
 * - NO hace SQL aquí: delega a los Repos/Modelos (PDO con sentencias preparadas).
 * - Devuelve estructuras EXACTAS que espera el frontend (JS).
 *
 * Convención de rango:
 *   $range = ['start' => 'YYYY-MM-DD', 'end' => 'YYYY-MM-DD']
 */
class DashboardService
{
    /** @var array<string,mixed> */
    private array $config;

    // Dependencias (repos)
    private SearchLogRepository    $searchRepo;
    private UnitStatsRepository    $unitRepo;
    private RouteRequestRepository $routeRepo;
    private KioskSessionRepository $kioskRepo;
    private CatalogRepository      $catalogRepo;
    private SecurityRepository     $securityRepo;
    private AnalyticsRepository    $analyticsRepo;

    public function __construct(
        array $config,
        SearchLogRepository    $searchRepo,
        UnitStatsRepository    $unitRepo,
        RouteRequestRepository $routeRepo,
        KioskSessionRepository $kioskRepo,
        CatalogRepository      $catalogRepo,
        SecurityRepository     $securityRepo,
        AnalyticsRepository    $analyticsRepo
    ) {
        $this->config       = $config;
        $this->searchRepo   = $searchRepo;
        $this->unitRepo     = $unitRepo;
        $this->routeRepo    = $routeRepo;
        $this->kioskRepo    = $kioskRepo;
        $this->catalogRepo  = $catalogRepo;
        $this->securityRepo = $securityRepo;
        $this->analyticsRepo= $analyticsRepo;
    }

    /* ============================================================
     * API PÚBLICA (11 MÉTODOS) - cada uno devuelve el payload
     * esperado por el JS del dashboard.
     * ============================================================ */

    /** Búsquedas totales + serie por día */
    public function getSearches(array $range): array
    {
        [$startDt, $endDt] = $this->expandDayBounds($range);

        $series = $this->searchRepo->countByDay($startDt, $endDt); // [['date'=>'YYYY-MM-DD','count'=>int], ...]
        $total  = array_sum(array_map(fn($r) => (int)($r['count'] ?? 0), $series));

        return [
            'count'  => (int)$total,
            'series' => array_map(fn($r) => [
                'date'  => (string)$r['date'],
                'count' => (int)$r['count'],
            ], $series),
        ];
    }

    /** % de búsquedas sin resultado */
    public function getNoResult(array $range): array
    {
        [$startDt, $endDt] = $this->expandDayBounds($range);

        $tot   = (int)$this->searchRepo->countTotal($startDt, $endDt);
        $noRes = (int)$this->searchRepo->countNoResult($startDt, $endDt);

        $pct = $tot > 0 ? round($noRes * 100.0 / $tot, 1) : 0.0;

        return ['no_result_pct' => $pct];
    }

    /** Top unidades más vistas/buscadas (top 5) */
    public function getTopUnits(array $range, int $limit = 5): array
    {
        [$startDt, $endDt] = $this->expandDayBounds($range);

        $items = $this->unitRepo->topUnits($startDt, $endDt, $limit); // [['name'=>..., 'hits'=>...], ...]
        return [
            'items' => array_map(fn($r) => [
                'name' => (string)$r['name'],
                'hits' => (int)$r['hits'],
            ], $items)
        ];
    }

    /** Embudo: Búsqueda → Unidad vista → Ruta solicitada */
    public function getFunnel(array $range): array
    {
        [$startDt, $endDt] = $this->expandDayBounds($range);

        $searches      = (int)$this->searchRepo->countTotal($startDt, $endDt);
        $unitViews     = (int)$this->unitRepo->countUnitViews($startDt, $endDt);
        $routeRequests = (int)$this->routeRepo->countRouteRequests($startDt, $endDt);

        return [
            'searches'       => $searches,
            'unit_views'     => $unitViews,
            'route_requests' => $routeRequests,
        ];
    }

    /** CTR: de búsqueda a selección (clic en resultado) */
    public function getCtr(array $range): array
    {
        [$startDt, $endDt] = $this->expandDayBounds($range);

        $withResults = (int)$this->searchRepo->countWithResults($startDt, $endDt);
        $resultClicks= (int)$this->searchRepo->countResultClicks($startDt, $endDt);

        $pct = $withResults > 0 ? round($resultClicks * 100.0 / $withResults, 1) : 0.0;
        return ['ctr_search_to_selection_pct' => $pct];
    }

    /** Total de solicitudes de ruta */
    public function getRouteRequests(array $range): array
    {
        [$startDt, $endDt] = $this->expandDayBounds($range);

        $count = (int)$this->routeRepo->countRouteRequests($startDt, $endDt);
        return ['count' => $count];
    }

    /** Sesiones de kiosko */
    public function getKioskSessions(array $range): array
    {
        [$startDt, $endDt] = $this->expandDayBounds($range);

        $count = (int)$this->kioskRepo->countSessions($startDt, $endDt);
        return ['count' => $count];
    }

    /** Higiene del catálogo (sin media, sin sinónimos, meta faltante) */
    public function getCatalogHygiene(array $range): array
    {
        // Puede no depender de fechas, pero mantenemos la firma por consistencia.
        $withoutMedia    = (int)$this->catalogRepo->countUnitsWithoutMedia();
        $withoutSynonyms = (int)$this->catalogRepo->countUnitsWithoutSynonyms();
        $missingMeta     = (int)$this->catalogRepo->countUnitsWithMissingMeta();

        return [
            'without_media'    => $withoutMedia,
            'without_synonyms' => $withoutSynonyms,
            'missing_meta'     => $missingMeta,
        ];
    }

    /** Seguridad: usuarios bloqueados (panel admin) */
    public function getSecurity(array $range): array
    {
        $locked = $this->securityRepo->getLockedUsers(); // [['email'=>..., 'locked_until'=>...], ...]
        return [
            'locked_count' => count($locked),
            'locked_list'  => array_map(fn($r) => [
                'email'        => (string)($r['email'] ?? ''),
                'locked_until' => (string)($r['locked_until'] ?? ''),
            ], $locked),
        ];
    }

    /** Uso por canal: kiosk vs web vs mobile */
public function getUsageByChannel(array $range): array
{
    [$startDt, $endDt] = $this->expandDayBounds($range);

    // El repo puede devolver:
    //  a) ['source' => 'kiosk','events'=>N]
    //  b) ['channel'=> 'kiosk','searches'=>N]
    $rows = $this->analyticsRepo->eventsBySource($startDt, $endDt);

    $items = array_map(function ($r) {
        $source = isset($r['source']) ? (string)$r['source']
                 : (isset($r['channel']) ? (string)$r['channel'] : '');
        $events = isset($r['events']) ? (int)$r['events']
                 : (isset($r['searches']) ? (int)$r['searches'] : 0);
        return ['source' => $source, 'events' => $events];
    }, is_array($rows) ? $rows : []);

    return ['items' => $items];
}


    /** Modo de entrada: voz vs texto */
    public function getInputMode(array $range): array
    {
        [$startDt, $endDt] = $this->expandDayBounds($range);

        $rows = $this->searchRepo->searchesByInputMode($startDt, $endDt); // [['mode'=>'text','searches'=>int], ...]
        return [
            'items' => array_map(fn($r) => [
                'mode'     => (string)$r['mode'],
                'searches' => (int)$r['searches'],
            ], $rows)
        ];
    }

    /* ============================================================
     * Helpers de dominio (pequeños, sin lógica de infraestructura)
     * ============================================================ */

    /**
     * Recibe ['start'=>'YYYY-MM-DD','end'=>'YYYY-MM-DD']
     * Devuelve ['YYYY-MM-DD 00:00:00','YYYY-MM-DD 23:59:59']
     */
    private function expandDayBounds(array $range): array
    {
        $start = (string)($range['start'] ?? '');
        $end   = (string)($range['end']   ?? $start);

        // Asumimos que el controlador ya validó formato YYYY-MM-DD
        $startDt = $start . ' 00:00:00';
        $endDt   = $end   . ' 23:59:59';

        // Sanidad: si el rango viene invertido, lo corregimos
        if ($startDt > $endDt) {
            [$startDt, $endDt] = [$endDt, $startDt];
        }
        return [$startDt, $endDt];
    }
}
