<?php
namespace App\Services\Admin;

use App\Models\Admin\AdminUserModel;

class AuthService
{
    private int $maxFails    = 5;   // intentos permitidos antes de bloquear
    private int $lockMinutes = 10;  // minutos de bloqueo

    private AdminUserModel $repo;

    public function __construct(private array $config) {
        if (session_status() !== PHP_SESSION_ACTIVE) {
           session_name('CampusMapAdmin');
           ini_set('session.gc_maxlifetime', 1800); // 30 minutos de inactividad
           session_start();
        }
        $this->repo = new AdminUserModel($config);
    }

    /** Intenta login con email+password. */
    public function login(string $email, string $password): array {
        $user = $this->repo->findByEmail($email);
        if (!$user) {
            return ['success'=>false, 'message'=>'Credenciales inválidas'];
        }

        if (!(int)$user['is_active']) {
            return ['success'=>false, 'message'=>'Usuario inactivo'];
        }

        // ¿Bloqueado aún?
        if (!empty($user['locked_until']) && strtotime($user['locked_until']) > time()) {
            $mins = (int)ceil((strtotime($user['locked_until']) - time()) / 60);
            return ['success'=>false, 'message'=>"Cuenta bloqueada. Intenta en ~{$mins} min."];
        }

        // Contraseña incorrecta
        if (!password_verify($password, $user['password_hash'])) {
            $this->repo->incrementFail((int)$user['id']);

            if (((int)$user['failed_attempts'] + 1) >= $this->maxFails) {
                $this->repo->lockFor((int)$user['id'], $this->lockMinutes);
                return ['success'=>false, 'message'=>"Demasiados intentos. Bloqueado {$this->lockMinutes} min."];
            }
            return ['success'=>false, 'message'=>'Credenciales inválidas'];
        }

        // Éxito → actualizar y setear sesión
        $this->repo->markLoginSuccess((int)$user['id']);
        
        session_regenerate_id(true);

        $_SESSION['admin_id']    = (int)$user['id'];
        $_SESSION['admin_pid']   = $user['public_id'];
        $_SESSION['admin_name']  = $user['name'];
        $_SESSION['admin_email'] = $user['email'];
        $_SESSION['admin_role']  = $user['role'];

        return ['success'=>true, 'message'=>'OK'];
    }

    public function logout(): void {
        $_SESSION = [];
        if (session_status() === PHP_SESSION_ACTIVE) session_destroy();
    }

    /** Devuelve datos básicos del usuario en sesión o false. */
    public function check(): array|false {
        if (session_status() !== PHP_SESSION_ACTIVE) {
            session_name('CampusMapAdmin');
            session_start();
        }
    
        //Control de inactividad
        $timeout = 1800; // 30 minutos
        $now = time();
    
        if (isset($_SESSION['last_activity'])) {
            $inactive = $now - $_SESSION['last_activity'];
            if ($inactive > $timeout) {
                // 🧹 Si supera el tiempo, cerrar sesión
                $this->logout();
                return false;
            }
        }
    
        // Actualiza hora de última actividad
        $_SESSION['last_activity'] = $now;
    
        // Devuelve datos del usuario si está logueado
        return !empty($_SESSION['admin_id']) ? [
            'id'        => $_SESSION['admin_id'],
            'public_id' => $_SESSION['admin_pid'] ?? null,
            'name'      => $_SESSION['admin_name'] ?? null,
            'email'     => $_SESSION['admin_email'] ?? null,
            'role'      => $_SESSION['admin_role'] ?? null,
        ] : false;
    }


    public static function role(): string {
        if (session_status() !== PHP_SESSION_ACTIVE) session_start();
        return $_SESSION['admin_role'] ?? '';
    }

    public static function isAdmin(): bool {
        return self::role() === 'admin';
    }

    public static function canViewStats(): bool {
        $r = self::role();
        return $r === 'admin' || $r === 'stats';
    }
}
