<?php
namespace App\Models;

use App\Core\Database;
use PDO;

class UnitRepository {
    private PDO $db;

    public function __construct(array $config) {
        $this->db = Database::pdo($config);
    }

    public function getBySlug(string $slug): ?array {
        $sql = "
            SELECT 
              u.*,
              a.slug AS area_slug,
              a.name AS area_name,
              (
                SELECT um.url 
                FROM unit_media um 
                WHERE um.unit_id = u.id AND um.type = 'image' 
                ORDER BY um.id ASC LIMIT 1
              ) AS image,
              (
                SELECT um.url 
                FROM unit_media um 
                WHERE um.unit_id = u.id AND um.type = 'video' 
                ORDER BY um.id ASC LIMIT 1
              ) AS video
            FROM units u
            LEFT JOIN areas a ON u.area_id = a.id
            WHERE u.slug = ? AND u.is_active = 1
            LIMIT 1
        ";
        $st = $this->db->prepare($sql);
        $st->execute([$slug]);
        return $st->fetch(PDO::FETCH_ASSOC) ?: null;
    }

    public function fulltext(string $term, int $limit = 5): array {
        $st = $this->db->prepare("
            SELECT id, name, alias, slug,
              MATCH(name, alias, description_short) AGAINST(? IN NATURAL LANGUAGE MODE) AS score
            FROM units
            WHERE is_active = 1 
              AND MATCH(name, alias, description_short) AGAINST(? IN NATURAL LANGUAGE MODE)
            ORDER BY score DESC
            LIMIT $limit
        ");
        $st->execute([$term, $term]);
        return $st->fetchAll(PDO::FETCH_ASSOC);
    }

    public function exactByNameOrAlias(string $t): ?array {
        $st = $this->db->prepare("
            SELECT * 
            FROM units 
            WHERE is_active = 1 
              AND (LOWER(name) = ? OR LOWER(alias) = ?) 
            LIMIT 1
        ");
        $st->execute([$t, $t]);
        return $st->fetch(PDO::FETCH_ASSOC) ?: null;
    }

    public function topByIds(array $ids): array {
        if (empty($ids)) return [];
        $in = implode(',', array_fill(0, count($ids), '?'));
        $st = $this->db->prepare("
            SELECT id, name, alias, slug 
            FROM units 
            WHERE id IN ($in) AND is_active = 1
        ");
        $st->execute($ids);
        return $st->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getAllWithAreas(): array {
        $sql = "
            SELECT 
              u.*, 
              a.slug AS area_slug, 
              a.name AS area_name,
              (
                SELECT um.url 
                FROM unit_media um 
                WHERE um.unit_id = u.id AND um.type = 'image' 
                ORDER BY um.id ASC LIMIT 1
              ) AS image,
              (
                SELECT um.url 
                FROM unit_media um 
                WHERE um.unit_id = u.id AND um.type = 'video' 
                ORDER BY um.id ASC LIMIT 1
              ) AS video
            FROM units u
            LEFT JOIN areas a ON u.area_id = a.id
            WHERE u.is_active = 1
            ORDER BY u.priority DESC
        ";
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getAllGroupedByArea(): array {
        $sql = "
            SELECT 
              a.name AS area_name, a.slug AS area_slug,
              u.id AS unit_id, u.name AS unit_name, u.slug AS unit_slug
            FROM units u
            JOIN areas a ON u.area_id = a.id
            WHERE u.is_active = 1
            ORDER BY a.name, u.name
        ";
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getByAreaSlug(string $slug): array {
        $sql = "
            SELECT 
              u.*, a.slug as area_slug, a.name as area_name,
              (
                SELECT um.url FROM unit_media um 
                WHERE um.unit_id = u.id AND um.type = 'image' 
                ORDER BY um.id ASC LIMIT 1
              ) AS image,
              (
                SELECT um.url FROM unit_media um 
                WHERE um.unit_id = u.id AND um.type = 'video' 
                ORDER BY um.id ASC LIMIT 1
              ) AS video
            FROM units u
            JOIN areas a ON u.area_id = a.id
            WHERE a.slug = ? AND u.is_active = 1
            ORDER BY u.priority DESC
        ";
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$slug]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function fetchAllUnits(int $limit, int $offset): array {
        $sql = "
            SELECT SQL_CALC_FOUND_ROWS 
              u.*, a.name AS area_name, a.slug AS area_slug,
              (SELECT url FROM unit_media WHERE unit_id = u.id AND type = 'image' LIMIT 1) AS image
            FROM units u
            JOIN areas a ON u.area_id = a.id
            WHERE u.is_active = 1
            ORDER BY u.priority DESC
            LIMIT :limit OFFSET :offset
        ";
        $st = $this->db->prepare($sql);
        $st->bindValue(':limit', $limit, PDO::PARAM_INT);
        $st->bindValue(':offset', $offset, PDO::PARAM_INT);
        $st->execute();
        $unitsa = $st->fetchAll(PDO::FETCH_ASSOC);

        $total = $this->db->query("SELECT FOUND_ROWS()")->fetchColumn();

        return ['unitsa' => $unitsa, 'total' => $total];
    }

    public function fetchUnitByIdOrSlug(?int $unitId, ?string $slug): ?array {
        $sql = "
            SELECT 
              u.*, 
              a.name AS area_name, 
              a.slug AS area_slug,
              m.url AS media_url, 
              m.type AS media_type
            FROM units u
            JOIN areas a ON u.area_id = a.id
            LEFT JOIN unit_media m ON m.unit_id = u.id
            WHERE u.is_active = 1
        ";

        $params = [];
        if ($unitId) {
            $sql .= " AND u.id = :id";
            $params[':id'] = $unitId;
        }
        if ($slug) {
            $sql .= " AND u.slug = :slug";
            $params[':slug'] = $slug;
        }

        $st = $this->db->prepare($sql);
        foreach ($params as $k => $v) {
            $st->bindValue($k, $v);
        }
        $st->execute();

        $rows = $st->fetchAll(PDO::FETCH_ASSOC);
        if (!$rows) {
            return null;
        }

        return $rows;
    }

    public function logStat(int $unitId, string $type, string $source): void {
        $st = $this->db->prepare("
            INSERT INTO unit_stats (unit_id, type, source) 
            VALUES (?, ?, ?)
        ");
        $st->execute([$unitId, $type, $source]);
    }
    
    public function getSummaryById(int $unitId): ?array {
    $sql = "SELECT 
                u.id, u.slug, u.name, u.alias,u.description_short,
                (SELECT url FROM unit_media WHERE unit_id = u.id AND type='image' LIMIT 1) AS image_url,
                (SELECT url FROM unit_media WHERE unit_id = u.id AND type='video' LIMIT 1) AS youtube_url
            FROM units u
            WHERE u.id = ?";
    $st = $this->db->prepare($sql);
    $st->execute([$unitId]);
    return $st->fetch(PDO::FETCH_ASSOC) ?: null;
}

}
