<?php
namespace App\Models\Admin;

use App\Core\Database;
use PDO;

/**
 * DashboardModelo
 * - Un solo modelo que implementa todos los repositorios usados por DashboardService.
 * - Usa App\Core\Database::pdo($config) para obtener PDO.
 * - Todas las consultas usan sentencias preparadas.
 *
 * IMPORTANTE: Ajusta las constantes TBL_* y nombres de columnas si tu esquema difiere.
 */
class DashboardModelo implements
    SearchLogRepository,
    UnitStatsRepository,
    RouteRequestRepository,
    KioskSessionRepository,
    CatalogRepository,
    SecurityRepository,
    AnalyticsRepository
{
    private PDO $db;

    public function __construct(array $config)
    {
        $this->db = Database::pdo($config);
    }

    /* ======= Tablas (ajusta a tu esquema real) ======= */
    private const TBL_SEARCH    = 'search_logs';      // cols: created_at, found (0/1), input_mode
    private const TBL_CLICKS    = 'result_clicks';    // cols: created_at
    private const TBL_UNITS     = 'units';            // cols: id, name, image_url, synonyms, meta_ok
    private const TBL_UNITVIEWS = 'unit_stats';       // cols: unit_id, created_at
    private const TBL_ROUTES    = 'route_requests';   // cols: created_at
    private const TBL_KIOSK     = 'kiosk_sessions';   // cols: started_at (o created_at)
    private const TBL_USERS     = 'admin_users';      // cols: email, locked_until
    private const TBL_EVENTS    = 'search_logs'; // cols: source, created_at

    /* ===================== SearchLogRepository ===================== */

    /** @return array<int,array{date:string,count:int}> */
    public function countByDay(string $startDt, string $endDt): array
    {
        $sql = "SELECT DATE(created_at) AS date, COUNT(*) AS count
                FROM ".self::TBL_SEARCH."
                WHERE created_at BETWEEN :start AND :end
                GROUP BY DATE(created_at)
                ORDER BY DATE(created_at) ASC";
        $st = $this->db->prepare($sql);
        $st->execute([':start'=>$startDt, ':end'=>$endDt]);
        $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
        return array_map(fn($r)=>['date'=>$r['date'],'count'=>(int)$r['count']], $rows);
    }

public function countTotal(string $startDt, string $endDt): int
{
    $st = $this->db->prepare("
        SELECT COUNT(*) AS c
        FROM search_logs
        WHERE created_at BETWEEN :start AND :end
    ");
    $st->execute([':start'=>$startDt, ':end'=>$endDt]);
    return (int)($st->fetch(PDO::FETCH_ASSOC)['c'] ?? 0);
}

public function countNoResult(string $startDt, string $endDt): int
{
    // Búsquedas sin resultado: matched_unit_id es NULL
    $st = $this->db->prepare(
        "SELECT COUNT(*) AS c 
         FROM ".self::TBL_SEARCH."
         WHERE created_at BETWEEN :start AND :end
         AND (matched_unit_id IS NULL OR matched_unit_id = 0)"
    );
    $st->execute([':start'=>$startDt, ':end'=>$endDt]);
    return (int)($st->fetch(PDO::FETCH_ASSOC)['c'] ?? 0);
}

public function countWithResults(string $startDt, string $endDt): int
{
    // Busquedas donde matched_unit_id NO es nulo
    $st = $this->db->prepare("
        SELECT COUNT(*) AS c
        FROM search_logs
        WHERE created_at BETWEEN :start AND :end
        AND matched_unit_id IS NOT NULL
    ");
    $st->execute([':start'=>$startDt, ':end'=>$endDt]);
    return (int)($st->fetch(PDO::FETCH_ASSOC)['c'] ?? 0);
}

public function countResultClicks(string $startDt, string $endDt): int
{
    // Si no existe tabla result_clicks, usamos unit_stats con type='play'
    $st = $this->db->prepare("
        SELECT COUNT(*) AS c
        FROM unit_stats
        WHERE created_at BETWEEN :start AND :end
        AND type = 'play'
    ");
    $st->execute([':start'=>$startDt, ':end'=>$endDt]);
    return (int)($st->fetch(PDO::FETCH_ASSOC)['c'] ?? 0);
}


    /** @return array<int,array{mode:string,searches:int}> */
    public function searchesByInputMode(string $startDt, string $endDt): array
    {
        $sql = "SELECT source AS mode, COUNT(*) AS searches
                FROM ".self::TBL_SEARCH."
                WHERE created_at BETWEEN :start AND :end
                GROUP BY source
                ORDER BY searches DESC";
        $st = $this->db->prepare($sql);
        $st->execute([':start'=>$startDt, ':end'=>$endDt]);
        $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
        return array_map(fn($r)=>['mode'=>$r['mode'],'searches'=>(int)$r['searches']], $rows);
    }

    /* ===================== UnitStatsRepository ===================== */

    /** @return array<int,array{name:string,hits:int}> */
    public function topUnits(string $startDt, string $endDt, int $limit): array
    {
        $sql = "SELECT u.name, COUNT(*) AS hits
                FROM ".self::TBL_UNITVIEWS." uv
                JOIN ".self::TBL_UNITS." u ON u.id = uv.unit_id
                WHERE uv.created_at BETWEEN :start AND :end
                GROUP BY uv.unit_id
                ORDER BY hits DESC
                LIMIT :limit";
        $st = $this->db->prepare($sql);
        $st->bindValue(':start', $startDt);
        $st->bindValue(':end',   $endDt);
        $st->bindValue(':limit', $limit, PDO::PARAM_INT);
        $st->execute();
        $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
        return array_map(fn($r)=>['name'=>$r['name'],'hits'=>(int)$r['hits']], $rows);
    }

public function countUnitViews(string $startDt, string $endDt): int
{
    $st = $this->db->prepare("
        SELECT COUNT(*) AS c
        FROM unit_stats
        WHERE created_at BETWEEN :start AND :end
        AND type = 'appearance'
    ");
    $st->execute([':start'=>$startDt, ':end'=>$endDt]);
    return (int)($st->fetch(PDO::FETCH_ASSOC)['c'] ?? 0);
}

    /* ===================== RouteRequestRepository ===================== */

public function countRouteRequests(string $startDt, string $endDt): int
{
    $st = $this->db->prepare("
        SELECT COUNT(*) AS c
        FROM unit_stats
        WHERE created_at BETWEEN :start AND :end
        AND type = 'play'
    ");
    $st->execute([':start'=>$startDt, ':end'=>$endDt]);
    return (int)($st->fetch(PDO::FETCH_ASSOC)['c'] ?? 0);
}

    /* ===================== KioskSessionRepository ===================== */

    public function countSessions(string $startDt, string $endDt): int
    {
        // Si tu columna en kiosk_sessions es created_at, cámbialo aquí.
        $st = $this->db->prepare(
            "SELECT COUNT(*) AS c FROM ".self::TBL_KIOSK."
             WHERE started_at BETWEEN :start AND :end"
        );
        $st->execute([':start'=>$startDt, ':end'=>$endDt]);
        return (int)($st->fetch(PDO::FETCH_ASSOC)['c'] ?? 0);
    }

    /* ===================== CatalogRepository ===================== */

    public function countUnitsWithoutMedia(): int
    {
        $row = $this->db->query(
            "SELECT COUNT(*) AS c FROM unit_media
             WHERE url IS NULL OR url = ''"
        )->fetch(PDO::FETCH_ASSOC);
        return (int)($row['c'] ?? 0);
    }

    public function countUnitsWithoutSynonyms(): int
    {
        $row = $this->db->query(
            "SELECT COUNT(*) AS c FROM unit_synonyms
             WHERE term IS NULL OR term = ''"
        )->fetch(PDO::FETCH_ASSOC);
        return (int)($row['c'] ?? 0);
    }

    public function countUnitsWithMissingMeta(): int
    {
        $row = $this->db->query(
            "SELECT COUNT(*) AS c FROM ".self::TBL_UNITS."
                     WHERE (description_short IS NULL OR description_short = '')
           OR (description_steps IS NULL OR description_steps = '')"
        )->fetch(PDO::FETCH_ASSOC);
        return (int)($row['c'] ?? 0);
    }

    /* ===================== SecurityRepository ===================== */

    /** @return array<int,array{email:string,locked_until:string}> */
    public function getLockedUsers(): array
    {
        $sql = "SELECT email, locked_until
                FROM ".self::TBL_USERS."
                WHERE locked_until IS NOT NULL AND locked_until > NOW()
                ORDER BY locked_until DESC";
        return $this->db->query($sql)->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /* ===================== AnalyticsRepository ===================== */

    /** @return array<int,array{source:string,events:int}> */
    public function eventsBySource(string $startDt, string $endDt): array
    {
    // total en search_logs
    $st = $this->db->prepare(
        "SELECT COUNT(*) AS c FROM search_logs
         WHERE created_at BETWEEN :start AND :end"
    );
    $st->execute([':start'=>$startDt, ':end'=>$endDt]);
    $total = (int)($st->fetch(PDO::FETCH_ASSOC)['c'] ?? 0);

    // kiosko en kiosk_stats por eventos de voz
    $st2 = $this->db->prepare(
         "SELECT COUNT(*) AS c FROM kiosk_sessions
         WHERE started_at BETWEEN :start AND :end"
    );
    $st2->execute([':start'=>$startDt, ':end'=>$endDt]);
    $kiosk = (int)($st2->fetch(PDO::FETCH_ASSOC)['c'] ?? 0);

    $web = ($total-$kiosk);

    return [
        ['channel' => 'kiosk', 'searches' => $kiosk],
        ['channel' => 'web',   'searches' => $web],
    ];
    }
}
