<?php
namespace App\Models\Admin;

use App\Core\Database;
use PDO;

class AdminUserModel {
    private PDO $db;

    public function __construct(array $config) {
        $this->db = Database::pdo($config);
    }

    /** Buscar por email (incluye UUID público como string legible). */
    public function findByEmail(string $email): ?array {
        $sql = "
            SELECT
              id,
              LOWER(CONCAT(
                SUBSTR(HEX(public_id),1,8),'-',
                SUBSTR(HEX(public_id),9,4),'-',
                SUBSTR(HEX(public_id),13,4),'-',
                SUBSTR(HEX(public_id),17,4),'-',
                SUBSTR(HEX(public_id),21)
              )) AS public_id,
              email, password_hash, name, role,
              is_active, failed_attempts, locked_until,
              last_login, created_at, updated_at
            FROM admin_users
            WHERE email = ?
            LIMIT 1
        ";
        $st = $this->db->prepare($sql);
        $st->execute([$email]);
        return $st->fetch(PDO::FETCH_ASSOC) ?: null;
    }

    /** Registrar login correcto: fecha + limpiar bloqueos/intentos. */
    public function markLoginSuccess(int $id): void {
        $st = $this->db->prepare("
            UPDATE admin_users
            SET last_login = NOW(),
                failed_attempts = 0,
                locked_until   = NULL
            WHERE id = ?
        ");
        $st->execute([$id]);
    }

    /** Incrementar intentos fallidos. */
    public function incrementFail(int $id): void {
        $st = $this->db->prepare("
            UPDATE admin_users
            SET failed_attempts = failed_attempts + 1
            WHERE id = ?
        ");
        $st->execute([$id]);
    }

    /** Bloquear por X minutos desde ahora y resetear contador. */
    public function lockFor(int $id, int $minutes): void {
        $st = $this->db->prepare("
            UPDATE admin_users
            SET locked_until   = DATE_ADD(NOW(), INTERVAL :mins MINUTE),
                failed_attempts = 0
            WHERE id = :id
        ");
        $st->bindValue(':mins', $minutes, PDO::PARAM_INT);
        $st->bindValue(':id', $id, PDO::PARAM_INT);
        $st->execute();
    }

    /** Quitar bloqueos/contador (sin tocar last_login). */
    public function resetFails(int $id): void {
        $st = $this->db->prepare("
            UPDATE admin_users
            SET failed_attempts = 0,
                locked_until   = NULL
            WHERE id = ?
        ");
        $st->execute([$id]);
    }

    /** Crear usuario admin (genera public_id en trigger). */
    public function create(string $email, string $hash, string $name, string $role = 'stats', int $active = 1): int {
        $st = $this->db->prepare("
            INSERT INTO admin_users (email, password_hash, name, role, is_active)
            VALUES (?, ?, ?, ?, ?)
        ");
        $st->execute([$email, $hash, $name, $role, $active]);
        return (int)$this->db->lastInsertId();
    }

    /** Activar / desactivar. */
    public function setActive(int $id, bool $active): void {
        $st = $this->db->prepare("UPDATE admin_users SET is_active = ? WHERE id = ?");
        $st->execute([$active ? 1 : 0, $id]);
    }

    /** (Opcional) Listado simple para administración. */
    public function listAll(): array {
        $sql = "
            SELECT
              id,
              LOWER(CONCAT(
                SUBSTR(HEX(public_id),1,8),'-',
                SUBSTR(HEX(public_id),9,4),'-',
                SUBSTR(HEX(public_id),13,4),'-',
                SUBSTR(HEX(public_id),17,4),'-',
                SUBSTR(HEX(public_id),21)
              )) AS public_id,
              email, name, role, is_active, last_login, created_at
            FROM admin_users
            ORDER BY created_at DESC
        ";
        return $this->db->query($sql)->fetchAll(PDO::FETCH_ASSOC);
    }
}
