<?php
namespace App\Controllers\Admin;

use App\Services\Admin\DashboardService;
use App\Models\Admin\DashboardModelo;

/**
 * Admin Stats API Controller (conectado a Service real)
 * Responde JSON para /admin/api/stats/*
 */
class StatsApiController
{
    private array $config;
    private DashboardService $service;

    private const TZ = 'America/El_Salvador';

    public function __construct(array $config)
    {
        $this->config = $config;

        // --- Modelo unificado que implementa todas las interfaces (repos) ---
        // OJO: DashboardModelo espera $config y resuelve PDO internamente con Database::pdo($config).
        $model = new DashboardModelo($config);

        // --- Service con sus dependencias (el mismo modelo para todas las interfaces) ---
        $this->service = new DashboardService(
            $config,
            $model, // SearchLogRepository
            $model, // UnitStatsRepository
            $model, // RouteRequestRepository
            $model, // KioskSessionRepository
            $model, // CatalogRepository
            $model, // SecurityRepository
            $model  // AnalyticsRepository
        );

        // Cabeceras comunes (por si index.php no las puso)
        if (!headers_sent()) {
            header('Content-Type: application/json; charset=utf-8');
            header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        }
    }

    /* ==================== Endpoints públicos ==================== */
    public function searches(array $q): void        { $this->handle($q, 'searches'); }
    public function noResult(array $q): void        { $this->handle($q, 'noResult'); }
    public function topUnits(array $q): void        { $this->handle($q, 'topUnits'); }
    public function funnel(array $q): void          { $this->handle($q, 'funnel'); }
    public function ctr(array $q): void             { $this->handle($q, 'ctr'); }
    public function routeRequests(array $q): void   { $this->handle($q, 'routeRequests'); }
    public function kioskSessions(array $q): void   { $this->handle($q, 'kioskSessions'); }
    public function catalogHygiene(array $q): void  { $this->handle($q, 'catalogHygiene'); }
    public function security(array $q): void        { $this->handle($q, 'security'); }
    public function usageByChannel(array $q): void  { $this->handle($q, 'usageByChannel'); }
    public function inputMode(array $q): void       { $this->handle($q, 'inputMode'); }
    // (Opcional) batch: /admin/api/stats/batch?metrics=a,b,c
    public function batch(array $q): void           { $this->handleBatch($q); }

    /* ==================== Core ==================== */

    private function handle(array $q, string $metric): void
    {
        $this->requireGET();
        $t0 = microtime(true);
        try {
            $range = $this->range($q);
            $data  = $this->resolveData($metric, $range);
            $this->jsonOk($data, $t0);
        } catch (\Throwable $e) {
            $this->jsonError($e->getMessage(), 400);
        }
    }

    private function handleBatch(array $q): void
    {
        $this->requireGET();
        $t0 = microtime(true);
        try {
            $range   = $this->range($q);
            $metrics = array_filter(array_map('trim', explode(',', (string)($q['metrics'] ?? ''))));
            if (!$metrics) {
                $this->jsonError('Parámetro "metrics" requerido (coma separada).', 400);
                return;
            }
            $out = [];
            foreach ($metrics as $slug) {
                $key = $this->normalize($slug);
                if (!$key) continue;
                $out[$slug] = $this->resolveData($key, $range);
            }
            $this->jsonOk($out, $t0);
        } catch (\Throwable $e) {
            $this->jsonError($e->getMessage(), 400);
        }
    }

    /**
     * Delegación al Service real.
     * $range = ['start'=>'YYYY-MM-DD', 'end'=>'YYYY-MM-DD']
     */
    private function resolveData(string $metric, array $range): array
    {
        return match ($metric) {
            'searches'        => $this->service->getSearches($range),
            'noResult'        => $this->service->getNoResult($range),
            'topUnits'        => $this->service->getTopUnits($range),
            'funnel'          => $this->service->getFunnel($range),
            'ctr'             => $this->service->getCtr($range),
            'routeRequests'   => $this->service->getRouteRequests($range),
            'kioskSessions'   => $this->service->getKioskSessions($range),
            'catalogHygiene'  => $this->service->getCatalogHygiene($range),
            'security'        => $this->service->getSecurity($range),
            'usageByChannel'  => $this->service->getUsageByChannel($range),
            'inputMode'       => $this->service->getInputMode($range),
            default           => throw new \RuntimeException('Métrica desconocida'),
        };
    }

    /* ==================== Utilidades ==================== */

    /** Solo permitimos GET para estos endpoints. */
    private function requireGET(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'GET') {
            http_response_code(405);
            echo json_encode(
                ['ok'=>false,'error'=>'Método no permitido'],
                JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES
            );
            exit;
        }
    }

    /** Normaliza el slug de la métrica desde la URL. */
    private function normalize(string $slug): ?string
    {
        $map = [
            'searches'         => 'searches',
            'no-result'        => 'noResult',
            'top-units'        => 'topUnits',
            'funnel'           => 'funnel',
            'ctr'              => 'ctr',
            'route-requests'   => 'routeRequests',
            'kiosk-sessions'   => 'kioskSessions',
            'catalog-hygiene'  => 'catalogHygiene',
            'security'         => 'security',
            'usage-by-channel' => 'usageByChannel',
            'input-mode'       => 'inputMode',
        ];
        return $map[$slug] ?? null;
    }

    /** Lee y valida ?start=YYYY-MM-DD&end=YYYY-MM-DD (TZ aware). Defaults: hoy. */
    private function range(array $q): array
    {
        $tz = new \DateTimeZone(self::TZ);
        $today = (new \DateTime('now', $tz))->format('Y-m-d');

        $start = $this->validDateOr((string)($q['start'] ?? $today), $today);
        $end   = $this->validDateOr((string)($q['end']   ?? $today), $today);

        if ($start > $end) {
            [$start, $end] = [$end, $start];
        }
        return ['start'=>$start, 'end'=>$end];
    }

    private function validDateOr(string $v, string $fallback): string
    {
        $d = \DateTime::createFromFormat('Y-m-d', $v);
        return ($d && $d->format('Y-m-d') === $v) ? $v : $fallback;
    }

    private function jsonOk(array $data, float $t0): void
    {
        $payload = [
            'ok'      => true,
            'data'    => $data,
            'took_ms' => (int)round((microtime(true) - $t0) * 1000),
        ];
        http_response_code(200);
        echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }

    private function jsonError(string $msg, int $code = 400): void
    {
        http_response_code($code);
        echo json_encode(
            ['ok'=>false,'error'=>$msg],
            JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES
        );
    }
}
